
#include <linux/cdev.h>
#include <linux/idr.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/mutex.h>
#include <linux/poll.h>
#include <linux/sched.h>
#include <linux/uaccess.h>
#include <linux/wait.h>

#include "u_f.h"
#include "ab_u_ptp.h"

#define AB_PTP_MINORS               2

#define AB_PTP_CTRL_REQ_LEN         64
#define AB_PTP_DATA_WRITE_REQ_NUM   1
#define AB_PTP_DATA_READ_REQ_NUM    4
#define AB_PTP_DATA_REQ_LEN         49152
#define AB_PTP_EVENT_READ_REQ_NUM   1
#define AB_PTP_EVENT_REQ_LEN        24

#define AB_PTP_DEV_NAME "ab_ptp_cdev"
#define AB_PTP_DEV_NODE "ab_ptp_comm"

#define AB_PTP_CTRL_CANCEL_REQUEST          0x64
#define AB_PTP_CTRL_GET_EXTENDED_EVENT_DATA 0x65
#define AB_PTP_CTRL_DEVICE_RESET_REQUEST    0x66
#define AB_PTP_CTRL_GET_DEVICE_STATUS       0x67

static int i_major, i_minors;
static struct class *ps_ptp_class;
static DEFINE_IDA(ab_ptp_ida);
static DEFINE_MUTEX(ab_ptp_ida_lock);

//===============================================================================
//========                        gadget structure                       ========
//===============================================================================

struct AB_F_PTP_REQ_LIST_ST {
    struct usb_request          *ps_req;
    unsigned int                ui_pos;
    struct list_head            s_list;
};

struct AB_F_PTP_READ_ST {
    spinlock_t                  s_lock;
    wait_queue_head_t           s_queue;
    struct list_head            s_completed_out_req;
};

struct AB_F_PTP_WRITE_ST {
    spinlock_t                  s_lock;
    wait_queue_head_t           s_queue;
    struct list_head            s_req_free;
};

struct AB_F_PTP_ST {
    // device
    char                        ac_name[32];
	int                         i_data_minor;
	int                         i_event_minor;
	struct cdev                 s_data_cdev;
	struct cdev                 s_event_cdev;
	struct usb_function         s_func;

    // ioctl
    struct AB_F_PTP_WRITE_ST    s_data_write;
    struct AB_F_PTP_READ_ST     s_data_read;
    struct AB_F_PTP_WRITE_ST    s_event_write;
    struct AB_F_PTP_READ_ST     s_event_read;   // does not support for now

    // ep
	struct usb_ep               *ps_in_ep;
	struct usb_ep               *ps_out_ep;
	struct usb_ep               *ps_intrpt_ep;

    // ep0
    struct usb_ep               *ps_ep0;
    struct usb_request          *ps_ctrl_req;
    __u8                        uc_req_out;
    __u8                        uc_req_type;
    __u16                       uw_req_len;

    //misc
    __u8                        uc_cancel;
};

static inline struct AB_F_PTP_ST *func_to_ab_ptp(struct usb_function *ps_func)
{
	return container_of(ps_func, struct AB_F_PTP_ST, s_func);
}

// For common device node
// this enumeration is the same as the one in ab_usb_ptp.c
typedef enum _AB_F_PTP_COMM_OP_E {
    AB_F_PTP_COMM_OP_UNKNOWN    = 0x00,
    AB_F_PTP_COMM_OP_GET_SPEED  = 0x01,
    AB_F_PTP_COMM_OP_TOTAL
} AB_F_PTP_COMM_OP_ET;

// this structure is the same as the one in ab_usb_ptp.c
typedef struct _AB_F_PTP_COMM_HEADER_S {
    __u32 ui_op;
    union {
        __u8 uc_speed;
        __u8 auc_reserved[28];
    } u_data;
} AB_F_PTP_COMM_HEADER_ST;

typedef struct _AB_F_PTP_COMM_CDEV_S {
    struct cdev     s_cdev;
    struct class    *ps_class;
    int             i_major;
    int             i_minor;
    __u8            uc_init;

    // get speed
    __u8            uc_get_speed;
    __u8            uc_speed;
} AB_F_PTP_COMM_CDEV_ST;

static AB_F_PTP_COMM_CDEV_ST *ab_f_ptp_comm = NULL;

//===============================================================================
//========                          descriptor                           ========
//===============================================================================

static struct usb_interface_descriptor s_ptp_intf_desc = {
    .bLength                = USB_DT_INTERFACE_SIZE,
    .bDescriptorType        = USB_DT_INTERFACE,
    // .bInterfaceNumber    = DYNAMIC
    .bAlternateSetting      = 0,
    .bNumEndpoints          = 3,
    .bInterfaceClass        = USB_CLASS_STILL_IMAGE,
    .bInterfaceSubClass     = 0x01, // still image capture device
    .bInterfaceProtocol     = 0x01, // bulk-only transport
    // .iInterface          = DYNAMIC
};

static struct usb_endpoint_descriptor s_ptp_fs_in_ep_desc = {
    .bLength            = USB_DT_ENDPOINT_SIZE,
    .bDescriptorType    = USB_DT_ENDPOINT,
    .bEndpointAddress   = USB_DIR_IN,
    .bmAttributes       = USB_ENDPOINT_XFER_BULK,
    .wMaxPacketSize     = cpu_to_le16(64U),
    .bInterval          = 0,
};

static struct usb_endpoint_descriptor s_ptp_hs_in_ep_desc = {
    .bLength            = USB_DT_ENDPOINT_SIZE,
    .bDescriptorType    = USB_DT_ENDPOINT,
    .bEndpointAddress   = USB_DIR_IN,
    .bmAttributes       = USB_ENDPOINT_XFER_BULK,
    .wMaxPacketSize     = cpu_to_le16(512U),
    .bInterval          = 0,
};

static struct usb_endpoint_descriptor s_ptp_ss_in_ep_desc = {
    .bLength            = USB_DT_ENDPOINT_SIZE,
    .bDescriptorType    = USB_DT_ENDPOINT,
    .bEndpointAddress   = USB_DIR_IN,
    .bmAttributes       = USB_ENDPOINT_XFER_BULK,
    .wMaxPacketSize     = cpu_to_le16(1024U),
    .bInterval          = 0,
};

static struct usb_ss_ep_comp_descriptor s_ptp_in_ep_comp_desc = {
    .bLength            = USB_DT_SS_EP_COMP_SIZE,
    .bDescriptorType    = USB_DT_SS_ENDPOINT_COMP,
    .bMaxBurst          = 15,
    .bmAttributes       = 0,
    .wBytesPerInterval  = 0,
};

static struct usb_endpoint_descriptor s_ptp_fs_out_ep_desc = {
    .bLength            = USB_DT_ENDPOINT_SIZE,
    .bDescriptorType    = USB_DT_ENDPOINT,
    .bEndpointAddress   = USB_DIR_OUT,
    .bmAttributes       = USB_ENDPOINT_XFER_BULK,
    .wMaxPacketSize     = cpu_to_le16(64U),
    .bInterval          = 0,
};

static struct usb_endpoint_descriptor s_ptp_hs_out_ep_desc = {
    .bLength            = USB_DT_ENDPOINT_SIZE,
    .bDescriptorType    = USB_DT_ENDPOINT,
    .bEndpointAddress   = USB_DIR_OUT,
    .bmAttributes       = USB_ENDPOINT_XFER_BULK,
    .wMaxPacketSize     = cpu_to_le16(512U),
    .bInterval          = 0,
};

static struct usb_endpoint_descriptor s_ptp_ss_out_ep_desc = {
    .bLength            = USB_DT_ENDPOINT_SIZE,
    .bDescriptorType    = USB_DT_ENDPOINT,
    .bEndpointAddress   = USB_DIR_OUT,
    .bmAttributes       = USB_ENDPOINT_XFER_BULK,
    .wMaxPacketSize     = cpu_to_le16(1024U),
    .bInterval          = 0,
};

static struct usb_ss_ep_comp_descriptor s_ptp_out_ep_comp_desc = {
    .bLength            = USB_DT_SS_EP_COMP_SIZE,
    .bDescriptorType    = USB_DT_SS_ENDPOINT_COMP,
    .bMaxBurst          = 15,
    .bmAttributes       = 0,
    .wBytesPerInterval  = 0,
};

static struct usb_endpoint_descriptor s_ptp_intrpt_ep_desc = {
    .bLength            = USB_DT_ENDPOINT_SIZE,
    .bDescriptorType    = USB_DT_ENDPOINT,
    .bEndpointAddress   = USB_DIR_IN,
    .bmAttributes       = USB_ENDPOINT_XFER_INT,
    .wMaxPacketSize     = cpu_to_le16(AB_PTP_EVENT_REQ_LEN),
    .bInterval          = 1,    // 125 us
};

static struct usb_ss_ep_comp_descriptor s_ptp_intrpt_ep_comp_desc = {
    .bLength            = USB_DT_SS_EP_COMP_SIZE,
    .bDescriptorType    = USB_DT_SS_ENDPOINT_COMP,
    .bMaxBurst          = 0,
    .bmAttributes       = 0,
    .wBytesPerInterval  = 0,
};

static struct usb_descriptor_header *aps_ptp_fs_desc[] = {
	(struct usb_descriptor_header *)&s_ptp_intf_desc, 
	(struct usb_descriptor_header *)&s_ptp_fs_in_ep_desc, 
	(struct usb_descriptor_header *)&s_ptp_fs_out_ep_desc, 
	(struct usb_descriptor_header *)&s_ptp_intrpt_ep_desc, 
	NULL,
};

static struct usb_descriptor_header *aps_ptp_hs_desc[] = {
	(struct usb_descriptor_header *)&s_ptp_intf_desc, 
	(struct usb_descriptor_header *)&s_ptp_hs_in_ep_desc, 
	(struct usb_descriptor_header *)&s_ptp_hs_out_ep_desc, 
	(struct usb_descriptor_header *)&s_ptp_intrpt_ep_desc, 
	NULL,
};

static struct usb_descriptor_header *aps_ptp_ss_desc[] = {
	(struct usb_descriptor_header *)&s_ptp_intf_desc, 
	(struct usb_descriptor_header *)&s_ptp_ss_in_ep_desc, 
	(struct usb_descriptor_header *)&s_ptp_in_ep_comp_desc, 
	(struct usb_descriptor_header *)&s_ptp_ss_out_ep_desc, 
	(struct usb_descriptor_header *)&s_ptp_out_ep_comp_desc, 
	(struct usb_descriptor_header *)&s_ptp_intrpt_ep_desc, 
	(struct usb_descriptor_header *)&s_ptp_intrpt_ep_comp_desc, 
	NULL,
};

//===============================================================================
//========                            string                             ========
//===============================================================================

#define AB_PTP_FUNC_STR_ID  0

static struct usb_string as_ptp_func_str_def[] = {
    [AB_PTP_FUNC_STR_ID].s = "AB PTP interface",
    {},
};

static struct usb_gadget_strings s_ptp_func_str_table = {
    .language   = 0x0409,
    .strings    = as_ptp_func_str_def,
};

static struct usb_gadget_strings *aps_ptp_func_str[] = {
    &s_ptp_func_str_table,
    NULL,
};

//===============================================================================
//========                           configfs                            ========
//===============================================================================

static inline struct AB_F_PTP_OPTS_ST *to_ab_f_ptp_opts(
    struct config_item *ps_item)
{
	return container_of(to_config_group(ps_item), struct AB_F_PTP_OPTS_ST, 
			            s_func_inst.group);
}

static void ab_ptp_attr_release(struct config_item *ps_item)
{
	struct AB_F_PTP_OPTS_ST *ps_opts = to_ab_f_ptp_opts(ps_item);

	usb_put_function_instance(&ps_opts->s_func_inst);
}

static struct configfs_item_operations ab_ptp_item_ops = {
	.release    = ab_ptp_attr_release,
};

static ssize_t ab_f_ptp_opts_dev0_show(
    struct config_item *ps_item, char *ps_page)
{
	struct AB_F_PTP_OPTS_ST *ps_opts = to_ab_f_ptp_opts(ps_item);

	return sprintf(ps_page, "%d:%d\n", i_major, ps_opts->i_data_minor);
}

CONFIGFS_ATTR_RO(ab_f_ptp_opts_, dev0);

static ssize_t ab_f_ptp_opts_dev1_show(
    struct config_item *ps_item, char *ps_page)
{
	struct AB_F_PTP_OPTS_ST *ps_opts = to_ab_f_ptp_opts(ps_item);

	return sprintf(ps_page, "%d:%d\n", i_major, ps_opts->i_event_minor);
}

CONFIGFS_ATTR_RO(ab_f_ptp_opts_, dev1);

static ssize_t ab_f_ptp_opts_name_show(
    struct config_item *ps_item, char *ps_page)
{
    struct AB_F_PTP_OPTS_ST *ps_opts = to_ab_f_ptp_opts(ps_item);

    return sprintf(ps_page, "%s", ps_opts->ac_name);
}

static ssize_t ab_f_ptp_opts_name_store(
    struct config_item *ps_item, const char *ps_page, size_t ss_len)
{
    struct AB_F_PTP_OPTS_ST *ps_opts = to_ab_f_ptp_opts(ps_item);

    if(ps_opts->i_refcnt)
        return -EBUSY;

    if(ss_len > PAGE_SIZE)
        return -ENOSPC;

    snprintf(ps_opts->ac_name, sizeof(ps_opts->ac_name), "%s", ps_page);
    return ss_len;
}

CONFIGFS_ATTR(ab_f_ptp_opts_, name);

static struct configfs_attribute *ab_ptp_attrs[] = {
	&ab_f_ptp_opts_attr_dev0,
	&ab_f_ptp_opts_attr_dev1,
	&ab_f_ptp_opts_attr_name,
	NULL,
};

static const struct config_item_type s_ptp_func_type = {
	.ct_item_ops    = &ab_ptp_item_ops,
	.ct_attrs       = ab_ptp_attrs,
	.ct_owner       = THIS_MODULE,
};

//===============================================================================
//========                        file operation                         ========
//===============================================================================

static void ab_ptp_data_write_req_complete(
    struct usb_ep *ps_ep, struct usb_request *ps_req);
static void ab_ptp_event_write_req_complete(
    struct usb_ep *ps_ep, struct usb_request *ps_req);

static __poll_t ab_f_ptp_data_poll(struct file *ps_fd, poll_table *ps_wait)
{
	struct AB_F_PTP_ST *ps_ptp = ps_fd->private_data;
    struct AB_F_PTP_WRITE_ST *ps_write = &ps_ptp->s_data_write;
    struct AB_F_PTP_READ_ST *ps_read = &ps_ptp->s_data_read;
	__poll_t e_ret = 0;

	poll_wait(ps_fd, &ps_read->s_queue, ps_wait);
	poll_wait(ps_fd, &ps_write->s_queue, ps_wait);

#define WRITE_COND  (!list_empty(&ps_write->s_req_free))
#define READ_COND   (!list_empty(&ps_read->s_completed_out_req))

    if(WRITE_COND)
    {
        e_ret |= EPOLLOUT | EPOLLWRNORM;
    }

    if(READ_COND)
    {
        e_ret |= EPOLLIN | EPOLLRDNORM;
    }

	return e_ret;
}

static ssize_t ab_f_ptp_data_read(
    struct file *ps_fd, char __user *pc_buf, size_t ss_cnt, loff_t *ps_ptr)
{
    struct AB_F_PTP_ST *ps_ptp = ps_fd->private_data;
    struct AB_F_PTP_READ_ST *ps_read = &ps_ptp->s_data_read;
    struct AB_F_PTP_REQ_LIST_ST *ps_list;
    struct usb_request *ps_req;
    unsigned long ul_flag;
    int i_ret;

	if (!ss_cnt)
		return 0;

	if(!access_ok(pc_buf, ss_cnt))
		return -EFAULT;

    spin_lock_irqsave(&ps_read->s_lock, ul_flag);
    while(!READ_COND)
    {
        spin_unlock_irqrestore(&ps_read->s_lock, ul_flag);

        if(ps_fd->f_flags & O_NONBLOCK)
            return -EAGAIN;

        if(wait_event_interruptible(ps_read->s_queue, READ_COND))
            return -ERESTARTSYS;

        spin_lock_irqsave(&ps_read->s_lock, ul_flag);
    }

    ps_list = list_first_entry(&ps_read->s_completed_out_req, 
                                struct AB_F_PTP_REQ_LIST_ST, s_list);

    ps_req = ps_list->ps_req;
    if(ss_cnt < ps_req->actual)
    {
        spin_unlock_irqrestore(&ps_read->s_lock, ul_flag);
        printk("[%s] buffer %ld < req %d\n",__func__, ss_cnt, ps_req->actual);
        ss_cnt = -EFAULT;
        return ss_cnt;
    }

    list_del(&ps_list->s_list);
    spin_unlock_irqrestore(&ps_read->s_lock, ul_flag);

    ss_cnt = copy_to_user(pc_buf, ps_req->buf, ps_req->actual);
    if(ss_cnt)
    {
        printk("[%s] copy fail: %ld -> %d\n",__func__, ss_cnt, ps_req->actual);
        spin_lock_irqsave(&ps_read->s_lock, ul_flag);
        list_add(&ps_list->s_list, &ps_read->s_completed_out_req);
        spin_unlock_irqrestore(&ps_read->s_lock, ul_flag);
        ss_cnt = -EFAULT;
        return ss_cnt;
    }
    ss_cnt = ps_req->actual;

    kfree(ps_list);
    ps_req->length = AB_PTP_DATA_REQ_LEN;
    i_ret = usb_ep_queue(ps_ptp->ps_out_ep, ps_req, GFP_KERNEL);
    if(i_ret < 0)
    {
        printk("[%s] enqueue fail: %d\n",__func__, i_ret);
        ss_cnt = i_ret;
    }

	return ss_cnt;
}

static ssize_t ab_f_ptp_data_write(
    struct file *ps_fd, const char __user *pc_buf, 
    size_t ss_cnt, loff_t *ps_ptr)
{
    struct AB_F_PTP_ST *ps_ptp = ps_fd->private_data;
    struct AB_F_PTP_WRITE_ST *ps_write = &ps_ptp->s_data_write;
    struct AB_F_PTP_REQ_LIST_ST *ps_list;
    struct usb_request *ps_req;
    unsigned long ul_flag;
	ssize_t ss_ret;

	if(!access_ok(pc_buf, ss_cnt))
		return -EFAULT;

    if(ps_ptp->uc_cancel)
    {
        ps_ptp->uc_cancel = 0;
		return -EFAULT;
    }

    spin_lock_irqsave(&ps_write->s_lock, ul_flag);
    while(!WRITE_COND)
    {
        spin_unlock_irqrestore(&ps_write->s_lock, ul_flag);

        if(ps_fd->f_flags & O_NONBLOCK)
            return -EAGAIN;

        if(wait_event_interruptible_exclusive(ps_write->s_queue, WRITE_COND))
            return -ERESTARTSYS;

        spin_lock_irqsave(&ps_write->s_lock, ul_flag);
    }

    ps_list = list_first_entry(&ps_write->s_req_free, 
                                struct AB_F_PTP_REQ_LIST_ST, s_list);
    list_del(&ps_list->s_list);
    ps_req = ps_list->ps_req;
    spin_unlock_irqrestore(&ps_write->s_lock, ul_flag);

    if(ps_ptp->uc_cancel)
    {
        ps_ptp->uc_cancel = 0;
        ss_ret = -ESHUTDOWN;
		goto err_end;
    }

    if(ss_cnt > AB_PTP_DATA_REQ_LEN)
    {
        printk("[%s] size error: %ld\n",__func__, ss_cnt);
        ss_ret = -EINVAL;
        goto err_end;
    }

    ss_ret = copy_from_user(ps_req->buf, pc_buf, ss_cnt);
    if(ss_ret)
    {
        printk("[%s] copy fail: %ld\n",__func__, ss_ret);
        ss_ret = -EINVAL;
        goto err_end;
    }

    ps_req->status = 0;
    ps_req->zero = 0;
    ps_req->length = ss_cnt;
    ps_req->complete = ab_ptp_data_write_req_complete;
    ps_req->context = ps_ptp;

    if(!ps_ptp->ps_in_ep->enabled)
    {
        printk("[%s] IN ep disable\n",__func__);
        ss_ret = -ESHUTDOWN;
        goto err_end;
    }

    if(ps_ptp->uc_cancel)
    {
        ps_ptp->uc_cancel = 0;
        ss_ret = -ESHUTDOWN;
		goto err_end;
    }

    ss_ret = usb_ep_queue(ps_ptp->ps_in_ep, ps_req, GFP_ATOMIC);
    if(ss_ret < 0)
    {
        printk("[%s] enqueue fail: %ld\n",__func__, ss_ret);
        goto err_end;
    }
    else
    {
        ss_ret = ss_cnt;
    }

    kfree(ps_list);

    if(ps_ptp->uc_cancel)
    {
        ps_ptp->uc_cancel = 0;
		return -ESHUTDOWN;
    }

    if(WRITE_COND)
        wake_up(&ps_write->s_queue);

    return ss_ret;

err_end:
    spin_lock_irqsave(&ps_write->s_lock, ul_flag);
    list_add_tail(&ps_list->s_list, &ps_write->s_req_free);
    spin_unlock_irqrestore(&ps_write->s_lock, ul_flag);
    wake_up(&ps_write->s_queue);
	return ss_ret;
}

#undef WRITE_COND
#undef READ_COND

static int ab_f_ptp_data_release(struct inode *ps_inode, struct file *ps_fd)
{
	ps_fd->private_data = NULL;
	return 0;
}

static int ab_f_ptp_data_open(struct inode *ps_inode, struct file *ps_fd)
{
	struct AB_F_PTP_ST *ps_ptp =
		container_of(ps_inode->i_cdev, struct AB_F_PTP_ST, s_data_cdev);

	ps_fd->private_data = ps_ptp;
	return 0;
}

static const struct file_operations ab_f_ptp_data_fops = {
	.owner		= THIS_MODULE,
	.open		= ab_f_ptp_data_open,
	.release	= ab_f_ptp_data_release,
	.write		= ab_f_ptp_data_write,
	.read		= ab_f_ptp_data_read,
	.poll		= ab_f_ptp_data_poll,
	.llseek		= noop_llseek,
};

static __poll_t ab_f_ptp_event_poll(struct file *ps_fd, poll_table *ps_wait)
{
	struct AB_F_PTP_ST *ps_ptp = ps_fd->private_data;
    struct AB_F_PTP_WRITE_ST *ps_write = &ps_ptp->s_event_write;
	__poll_t e_ret = 0;

	poll_wait(ps_fd, &ps_write->s_queue, ps_wait);

#define WRITE_COND  (!list_empty(&ps_write->s_req_free))

    if(WRITE_COND)
    {
        e_ret |= EPOLLOUT | EPOLLWRNORM;
    }

	return e_ret;
}

static ssize_t ab_f_ptp_event_read(
    struct file *ps_fd, char __user *pc_buf, size_t ss_cnt, loff_t *ps_ptr)
{
/*
 *  Not support for now
 */
	return -EFAULT;
}

static ssize_t ab_f_ptp_event_write(
    struct file *ps_fd, const char __user *pc_buf, 
    size_t ss_cnt, loff_t *ps_ptr)
{
    struct AB_F_PTP_ST *ps_ptp = ps_fd->private_data;
    struct AB_F_PTP_WRITE_ST *ps_write = &ps_ptp->s_event_write;
    struct AB_F_PTP_REQ_LIST_ST *ps_list;
    struct usb_request *ps_req;
    unsigned long ul_flag;
	ssize_t ss_ret;

	if(!access_ok(pc_buf, ss_cnt))
		return -EFAULT;

    spin_lock_irqsave(&ps_write->s_lock, ul_flag);
    while(!WRITE_COND)
    {
        spin_unlock_irqrestore(&ps_write->s_lock, ul_flag);

        if(ps_fd->f_flags & O_NONBLOCK)
            return -EAGAIN;

        if(wait_event_interruptible_exclusive(ps_write->s_queue, WRITE_COND))
            return -ERESTARTSYS;

        spin_lock_irqsave(&ps_write->s_lock, ul_flag);
    }

    ps_list = list_first_entry(&ps_write->s_req_free, 
                                struct AB_F_PTP_REQ_LIST_ST, s_list);
    list_del(&ps_list->s_list);
    ps_req = ps_list->ps_req;
    ss_cnt = min_t(unsigned, ss_cnt, AB_PTP_EVENT_REQ_LEN);
    spin_unlock_irqrestore(&ps_write->s_lock, ul_flag);

    ss_ret = copy_from_user(ps_req->buf, pc_buf, ss_cnt);
    if(ss_ret)
    {
        printk("[%s] copy fail: %ld\n",__func__, ss_ret);
        ss_ret = -EINVAL;
        goto err_end;
    }

    ps_req->status = 0;
    ps_req->zero = 0;
    ps_req->length = ss_cnt;
    ps_req->complete = ab_ptp_event_write_req_complete;
    ps_req->context = ps_ptp;

    if(!ps_ptp->ps_intrpt_ep->enabled)
    {
        printk("[%s] INTRP ep disable\n",__func__);
        ss_ret = -ESHUTDOWN;
        goto err_end;
    }

    ss_ret = usb_ep_queue(ps_ptp->ps_intrpt_ep, ps_req, GFP_ATOMIC);
    if(ss_ret < 0)
    {
        printk("[%s] enqueue fail: %ld\n",__func__, ss_ret);
        goto err_end;
    }
    else
    {
        ss_ret = ss_cnt;
    }

    kfree(ps_list);
    if(WRITE_COND)
        wake_up(&ps_write->s_queue);

    return ss_ret;

err_end:
    spin_lock_irqsave(&ps_write->s_lock, ul_flag);
    list_add_tail(&ps_list->s_list, &ps_write->s_req_free);
    spin_unlock_irqrestore(&ps_write->s_lock, ul_flag);
    wake_up(&ps_write->s_queue);
	return ss_ret;
}

#undef WRITE_COND

static int ab_f_ptp_event_release(struct inode *ps_inode, struct file *ps_fd)
{
	ps_fd->private_data = NULL;
	return 0;
}

static int ab_f_ptp_event_open(struct inode *ps_inode, struct file *ps_fd)
{
	struct AB_F_PTP_ST *ps_ptp =
		container_of(ps_inode->i_cdev, struct AB_F_PTP_ST, s_event_cdev);

	ps_fd->private_data = ps_ptp;
	return 0;
}

static const struct file_operations ab_f_ptp_event_fops = {
	.owner		= THIS_MODULE,
	.open		= ab_f_ptp_event_open,
	.release	= ab_f_ptp_event_release,
	.write		= ab_f_ptp_event_write,
	.read		= ab_f_ptp_event_read,
	.poll		= ab_f_ptp_event_poll,
	.llseek		= noop_llseek,
};


// For common device node
static ssize_t ab_f_ptp_comm_read(
    struct file *ps_fd, char __user *pc_buf, size_t ss_cnt, loff_t *ps_ptr)
{
    AB_F_PTP_COMM_CDEV_ST *ps_ptp_info = ps_fd->private_data;
    AB_F_PTP_COMM_HEADER_ST s_header = {0};
    ssize_t ss_ret, ss_tmp;

    ss_tmp = sizeof(s_header);
    if (ss_cnt < ss_tmp)
		return -EINVAL;

	if(!access_ok(pc_buf, ss_cnt))
		return -EACCES;

    ss_ret = copy_from_user(&s_header, pc_buf, sizeof(s_header));
    if(ss_ret)
        return -EFAULT;

    switch(s_header.ui_op)
    {
        case AB_F_PTP_COMM_OP_GET_SPEED:
            if(!ps_ptp_info->uc_get_speed)
                return -EAGAIN;

            s_header.u_data.uc_speed = ps_ptp_info->uc_speed;
            ss_ret = copy_to_user(pc_buf, &s_header, sizeof(s_header));
            if(ss_ret)
                return -EFAULT;

            break;
        default:
            return -EPERM;
    }

	return 0;
}

static ssize_t ab_f_ptp_comm_write(
    struct file *ps_fd, const char __user *pc_buf, 
    size_t ss_cnt, loff_t *ps_ptr)
{
    return 0;
}

static int ab_f_ptp_comm_release(struct inode *ps_inode, struct file *ps_fd)
{
    ps_fd->private_data = NULL;
	return 0;
}

static int ab_f_ptp_comm_open(struct inode *ps_inode, struct file *ps_fd)
{
	AB_F_PTP_COMM_CDEV_ST *ps_ptp_info =
		container_of(ps_inode->i_cdev, AB_F_PTP_COMM_CDEV_ST, s_cdev);

	ps_fd->private_data = ps_ptp_info;
	return 0;
}

static const struct file_operations ab_f_ptp_comm_fops = {
	.owner		= THIS_MODULE,
	.open		= ab_f_ptp_comm_open,
	.release	= ab_f_ptp_comm_release,
	.write		= ab_f_ptp_comm_write,
	.read		= ab_f_ptp_comm_read,
	.llseek		= noop_llseek,
};

//===============================================================================
//========                         usb function                          ========
//===============================================================================

// For common device node
static void ab_ptp_comm_set_speed(struct usb_function *ps_func, __u8 uc_en)
{
    struct usb_composite_dev *ps_cdev;

    if(!ab_f_ptp_comm)
        return;

    if(!uc_en)
    {
        ab_f_ptp_comm->uc_get_speed = 0;
        return;
    }

    ps_cdev = ps_func->config->cdev;
    ab_f_ptp_comm->uc_get_speed = 1;
    ab_f_ptp_comm->uc_speed = ps_cdev->gadget->speed;
}

static void ab_ptp_comm_make_class(void)
{
    dev_t e_dev;
    int i_ret;

    if(ab_f_ptp_comm)
        return;

    ab_f_ptp_comm = kzalloc(sizeof(AB_F_PTP_COMM_CDEV_ST), GFP_KERNEL);
    if(!ab_f_ptp_comm)
    {
        printk("[%s] alloc fail\n",__func__);
        ab_f_ptp_comm = NULL;
        return;
    }

    ab_f_ptp_comm->ps_class = class_create(THIS_MODULE, AB_PTP_DEV_NAME);
    if(IS_ERR(ab_f_ptp_comm->ps_class))
    {
        printk("[%s] create class fail\n",__func__);
        goto err;
    }

    i_ret = alloc_chrdev_region(&e_dev, 0, 1, AB_PTP_DEV_NAME);
    if(i_ret)
    {
        printk("[%s] alloc fail\n",__func__);
        class_destroy(ab_f_ptp_comm->ps_class);
        goto err;
    }

    ab_f_ptp_comm->i_major = MAJOR(e_dev);
    ab_f_ptp_comm->i_minor = MINOR(e_dev);
    return;

err:
    kfree(ab_f_ptp_comm);
    ab_f_ptp_comm = NULL;
}

static void ab_ptp_comm_remove_class(void)
{
    dev_t e_dev;

    if(!ab_f_ptp_comm)
        return;

    e_dev = MKDEV(ab_f_ptp_comm->i_major, ab_f_ptp_comm->i_minor);
    unregister_chrdev_region(e_dev, 1);
    class_destroy(ab_f_ptp_comm->ps_class);
    kfree(ab_f_ptp_comm);
    ab_f_ptp_comm = NULL;
}

static void ab_ptp_comm_make_cdev(void)
{
    struct device *ps_device;
    dev_t e_dev;
    int i_ret;

    if(!ab_f_ptp_comm)
        return;

    if(ab_f_ptp_comm->uc_init)
        return;

    e_dev = MKDEV(ab_f_ptp_comm->i_major, ab_f_ptp_comm->i_minor);
    cdev_init(&ab_f_ptp_comm->s_cdev, &ab_f_ptp_comm_fops);
    i_ret = cdev_add(&ab_f_ptp_comm->s_cdev, e_dev, 1);
    if(i_ret)
    {
        printk("[%s] add fail\n",__func__);
        ab_ptp_comm_remove_class();
        return;
    }

    ps_device = device_create(ab_f_ptp_comm->ps_class, NULL, 
                                e_dev, NULL, AB_PTP_DEV_NODE);
    if(IS_ERR(ps_device))
    {
        printk("[%s] create device fail\n",__func__);
        ab_ptp_comm_remove_class();
        cdev_del(&ab_f_ptp_comm->s_cdev);
        return;
    }

    ab_f_ptp_comm->uc_init = 1;
}

static void ab_ptp_comm_remove_cdev(void)
{
    dev_t e_dev;

    if(!ab_f_ptp_comm)
        return;

    if(!ab_f_ptp_comm->uc_init)
        return;

    e_dev = MKDEV(ab_f_ptp_comm->i_major, ab_f_ptp_comm->i_minor);
    device_destroy(ab_f_ptp_comm->ps_class, e_dev);
    cdev_del(&ab_f_ptp_comm->s_cdev);

    ab_f_ptp_comm->uc_init = 0;
}

// PTP function
static void ab_ptp_ctrl_req_complete(
    struct usb_ep *ps_ep, struct usb_request *ps_req)
{
    struct AB_F_PTP_ST *ps_ptp = ps_req->context;

    if(!ps_ptp->uc_req_out)
        return;

    ps_ptp->uc_req_out = 0;
    switch(ps_ptp->uc_req_type)
    {
        case AB_PTP_CTRL_CANCEL_REQUEST:
            if(ps_req->actual != ps_ptp->uw_req_len)
            {
                printk("[%s] size error: %d -> %d\n",__func__, 
                        ps_req->actual, ps_ptp->uw_req_len);
            }
            break;
        default:
            printk("[%s] request: %x\n",__func__, ps_ptp->uc_req_type);
            break;
    }
}

static void ab_ptp_data_write_req_complete(
    struct usb_ep *ps_ep, struct usb_request *ps_req)
{
    struct AB_F_PTP_ST *ps_ptp = ps_req->context;
    struct AB_F_PTP_WRITE_ST *ps_write = &ps_ptp->s_data_write;
    struct AB_F_PTP_REQ_LIST_ST *ps_list = NULL;
    unsigned long ul_flag;

    if(ps_req->status)
        printk("[%s] req error: %d\n",__func__, ps_req->status);

    ps_list = kzalloc(sizeof(struct AB_F_PTP_REQ_LIST_ST), GFP_ATOMIC);
    if(!ps_list)
    {
        printk("[%s] alloc list data IN fail\n",__func__);
		free_ep_req(ps_ep, ps_req);
    }

    ps_list->ps_req = ps_req;
    spin_lock_irqsave(&ps_write->s_lock, ul_flag);
    list_add_tail(&ps_list->s_list, &ps_write->s_req_free);
    spin_unlock_irqrestore(&ps_write->s_lock, ul_flag);
    wake_up(&ps_write->s_queue);
}

static void ab_ptp_data_read_req_complete(
    struct usb_ep *ps_ep, struct usb_request *ps_req)
{
    struct AB_F_PTP_ST *ps_ptp = ps_req->context;
    struct AB_F_PTP_READ_ST *ps_read = &ps_ptp->s_data_read;
    struct AB_F_PTP_REQ_LIST_ST *ps_list = NULL;
    unsigned long ul_flag;

    switch(ps_req->status)
    {
        case 0:
            ps_list = kzalloc(sizeof(struct AB_F_PTP_REQ_LIST_ST), GFP_ATOMIC);
            if(!ps_list)
            {
                printk("[%s] alloc list data OUT fail\n",__func__);
				break;
            }

            ps_list->ps_req = ps_req;
            spin_lock_irqsave(&ps_read->s_lock, ul_flag);
            list_add_tail(&ps_list->s_list, &ps_read->s_completed_out_req);
            spin_unlock_irqrestore(&ps_read->s_lock, ul_flag);
            wake_up(&ps_read->s_queue);
            break;
        default:
        case -ECONNABORTED:
        case -ECONNRESET:
        case -ESHUTDOWN:
            free_ep_req(ps_ep, ps_req);
            break;
    }
}

static void ab_ptp_event_write_req_complete(
    struct usb_ep *ps_ep, struct usb_request *ps_req)
{
    struct AB_F_PTP_ST *ps_ptp = ps_req->context;
    struct AB_F_PTP_WRITE_ST *ps_write = &ps_ptp->s_event_write;
    struct AB_F_PTP_REQ_LIST_ST *ps_list = NULL;
    unsigned long ul_flag;

    if(ps_req->status)
        printk("[%s] req error: %d\n",__func__, ps_req->status);

    ps_list = kzalloc(sizeof(struct AB_F_PTP_REQ_LIST_ST), GFP_ATOMIC);
    if(!ps_list)
    {
        printk("[%s] alloc list data IN fail\n",__func__);
		free_ep_req(ps_ep, ps_req);
    }

    ps_list->ps_req = ps_req;
    spin_lock_irqsave(&ps_write->s_lock, ul_flag);
    list_add_tail(&ps_list->s_list, &ps_write->s_req_free);
    spin_unlock_irqrestore(&ps_write->s_lock, ul_flag);
    wake_up(&ps_write->s_queue);
}

static int ab_ptp_setup(
    struct usb_function *ps_func, const struct usb_ctrlrequest *ps_ctrl)
{
    struct AB_F_PTP_ST *ps_ptp;
    struct usb_request *ps_req;
    __u16 uw_tmp;
    int i_ret;

    if(((ps_ctrl->bRequestType & USB_TYPE_MASK) != USB_TYPE_CLASS) || 
        ((ps_ctrl->bRequestType & USB_RECIP_MASK) != USB_RECIP_INTERFACE))
    {
        printk("[%s] invalid request type\n",__func__);
        return -EINVAL;
    }

    if(le16_to_cpu(ps_ctrl->wLength) > AB_PTP_CTRL_REQ_LEN)
    {
        printk("[%s] request too large\n",__func__);
        return -EINVAL;
    }

    ps_ptp = container_of(ps_func, struct AB_F_PTP_ST, s_func);
    ps_req = ps_ptp->ps_ctrl_req;
    if(!ps_req)
    {
        printk("[%s] ctrl req not ready\n",__func__);
        return -EINVAL;
    }

    switch(ps_ctrl->bRequest)
    {
        case AB_PTP_CTRL_CANCEL_REQUEST:
            if(le16_to_cpu(ps_ctrl->wLength) != 6)
            {
                printk("[%s] CANCEL_REQUEST: invalid length: %d\n",__func__, 
                        le16_to_cpu(ps_ctrl->wLength));
                return -EINVAL;
            }

            ps_ptp->uc_cancel = 1;

            ps_req->length = le16_to_cpu(ps_ctrl->wLength);
            ps_req->zero = 0;
            memset(ps_req->buf, 0, ps_req->length);
            i_ret = usb_ep_queue(ps_ptp->ps_ep0, ps_req, GFP_ATOMIC);
            if(i_ret)
            {
                printk("[%s] CANCEL_REQUEST: queue fail: %d\n",__func__, i_ret);
                return -EINVAL;
            }

            ps_ptp->uc_req_out = 1;
            ps_ptp->uc_req_type = AB_PTP_CTRL_CANCEL_REQUEST;
            ps_ptp->uw_req_len = le16_to_cpu(ps_ctrl->wLength);
            break;
        case AB_PTP_CTRL_GET_DEVICE_STATUS:
            uw_tmp = 4;
            ps_req->length = uw_tmp;
            memcpy(ps_req->buf, &uw_tmp, 2);
            uw_tmp = 0x2001;
            memcpy(ps_req->buf + 2, &uw_tmp, 2);
            i_ret = usb_ep_queue(ps_ptp->ps_ep0, ps_req, GFP_ATOMIC);
            if(i_ret)
            {
                printk("[%s] GET_DEVICE_STATUS: queue fail: %d\n",__func__, 
                        i_ret);
                return -EINVAL;
            }
            break;
        case AB_PTP_CTRL_GET_EXTENDED_EVENT_DATA:
        case AB_PTP_CTRL_DEVICE_RESET_REQUEST:
            // not handle for now
        default:
            printk("[%s] request: %x\n",__func__, ps_ctrl->bRequest);
            return -EINVAL;
    }

    return 0;
}

static void ab_ptp_disable(struct usb_function *ps_func)
{
    struct AB_F_PTP_ST *ps_ptp = func_to_ab_ptp(ps_func);
    struct AB_F_PTP_REQ_LIST_ST *ps_list, *ps_next;
    struct AB_F_PTP_WRITE_ST *ps_write;
    struct AB_F_PTP_READ_ST *ps_read;
    unsigned long ul_flag;

    INFO(ps_func->config->cdev, "ab_ptp_disable\n");

    ab_ptp_comm_set_speed(ps_func, 0);

    if(ps_ptp->ps_in_ep)
    {
        usb_ep_disable(ps_ptp->ps_in_ep);

        ps_write = &ps_ptp->s_data_write;
        spin_lock_irqsave(&ps_write->s_lock, ul_flag);
        list_for_each_entry_safe(ps_list, ps_next, 
                                &ps_write->s_req_free, s_list)
        {
            free_ep_req(ps_ptp->ps_in_ep, ps_list->ps_req);
            list_del(&ps_list->s_list);
            kfree(ps_list);
        }

        spin_unlock_irqrestore(&ps_write->s_lock, ul_flag);
    }

    if(ps_ptp->ps_out_ep)
    {
        usb_ep_disable(ps_ptp->ps_out_ep);

        ps_read = &ps_ptp->s_data_read;
        spin_lock_irqsave(&ps_read->s_lock, ul_flag);
        list_for_each_entry_safe(ps_list, ps_next, 
                                &ps_read->s_completed_out_req, s_list)
        {
            free_ep_req(ps_ptp->ps_in_ep, ps_list->ps_req);
            list_del(&ps_list->s_list);
            kfree(ps_list);
        }
        spin_unlock_irqrestore(&ps_read->s_lock, ul_flag);
    }

    if(ps_ptp->ps_intrpt_ep)
    {
        usb_ep_disable(ps_ptp->ps_intrpt_ep);

        ps_write = &ps_ptp->s_event_write;
        spin_lock_irqsave(&ps_write->s_lock, ul_flag);
        list_for_each_entry_safe(ps_list, ps_next, 
                                &ps_write->s_req_free, s_list)
        {
            free_ep_req(ps_ptp->ps_in_ep, ps_list->ps_req);
            list_del(&ps_list->s_list);
            kfree(ps_list);
        }

        spin_unlock_irqrestore(&ps_write->s_lock, ul_flag);
    }
}

static int ab_ptp_set_alt(
    struct usb_function *ps_func, unsigned ui_intf, unsigned ui_alt)
{
    struct usb_composite_dev *ps_cdev = ps_func->config->cdev;
    struct AB_F_PTP_ST *ps_ptp = func_to_ab_ptp(ps_func);
    struct AB_F_PTP_WRITE_ST *ps_write;
    struct AB_F_PTP_READ_ST *ps_read;
    struct AB_F_PTP_REQ_LIST_ST *ps_list, *ps_next;
    struct usb_request *ps_req;
    int i_cnt = 0;
    int i_ret = 0;

    INFO(ps_cdev, "ab_ptp_set_alt: (%d,%d)\n", ui_intf, ui_alt);

    ab_ptp_comm_set_speed(ps_func, 1);

    if(ps_ptp->ps_in_ep)
    {
        usb_ep_disable(ps_ptp->ps_in_ep);
        i_ret = config_ep_by_speed(ps_cdev->gadget, ps_func, ps_ptp->ps_in_ep);
        if(i_ret)
        {
            printk("[%s] config_ep_by_speed data IN fail\n",__func__);
            goto fail;
        }

        i_ret = usb_ep_enable(ps_ptp->ps_in_ep);
        if(i_ret < 0)
        {
            printk("[%s] usb_ep_enable data IN fail\n",__func__);
            goto fail;
        }
        ps_ptp->ps_in_ep->driver_data = ps_ptp;

        ps_write = &ps_ptp->s_data_write;
        for(i_cnt = 0; i_cnt < AB_PTP_DATA_WRITE_REQ_NUM; i_cnt++)
        {
            ps_req = alloc_ep_req(ps_ptp->ps_in_ep, AB_PTP_DATA_REQ_LEN);
            if(!ps_req)
            {
                printk("[%s] alloc_ep_req data IN %d fail\n",__func__, i_cnt);
                i_ret = -ENOMEM;
                goto clean_up_data_in;
            }

            ps_list = NULL;
            ps_list = kzalloc(sizeof(struct AB_F_PTP_REQ_LIST_ST), GFP_ATOMIC);
            if(!ps_list)
            {
                printk("[%s] alloc list data IN %d fail\n",__func__, i_cnt);
                free_ep_req(ps_ptp->ps_in_ep, ps_req);
                i_ret = -ENOMEM;
                goto clean_up_data_in;
            }

            ps_list->ps_req = ps_req;
            list_add_tail(&ps_list->s_list, &ps_write->s_req_free);
        }
    }

    ps_read = &ps_ptp->s_data_read;
    if(ps_ptp->ps_out_ep)
    {
        usb_ep_disable(ps_ptp->ps_out_ep);
        i_ret = config_ep_by_speed(ps_cdev->gadget, ps_func, ps_ptp->ps_out_ep);
        if(i_ret)
        {
            printk("[%s] config_ep_by_speed data OUT fail\n",__func__);
            goto clean_up_data_in;
        }

        i_ret = usb_ep_enable(ps_ptp->ps_out_ep);
        if(i_ret < 0)
        {
            printk("[%s] usb_ep_enable data OUT fail\n",__func__);
            goto clean_up_data_in;
        }
        ps_ptp->ps_out_ep->driver_data = ps_ptp;

        ps_read = &ps_ptp->s_data_read;
        for(i_cnt = 0; i_cnt < AB_PTP_DATA_READ_REQ_NUM; i_cnt++)
        {
            ps_req = alloc_ep_req(ps_ptp->ps_out_ep, AB_PTP_DATA_REQ_LEN);
            if(!ps_req)
            {
                printk("[%s] alloc_ep_req data OUT %d fail\n",__func__, i_cnt);
                i_ret = -ENOMEM;
                goto clean_up_data_out;
            }

            ps_req->complete = ab_ptp_data_read_req_complete;
            ps_req->context = ps_ptp;
            i_ret = usb_ep_queue(ps_ptp->ps_out_ep, ps_req, GFP_ATOMIC);
            if(i_ret)
            {
                printk("[%s] %s queue req --> %d\n",__func__, 
                        ps_ptp->ps_out_ep->name, i_ret);
                list_del(&ps_list->s_list);
                kfree(ps_list);
				free_ep_req(ps_ptp->ps_out_ep, ps_req);
                goto clean_up_data_out;
            }
        }
    }

    if(ps_ptp->ps_intrpt_ep)
    {
        usb_ep_disable(ps_ptp->ps_intrpt_ep);
        i_ret = config_ep_by_speed(ps_cdev->gadget, ps_func, 
                                    ps_ptp->ps_intrpt_ep);
        if(i_ret)
        {
            printk("[%s] config_ep_by_speed event IN fail\n",__func__);
            goto clean_up_data_out;
        }

        i_ret = usb_ep_enable(ps_ptp->ps_intrpt_ep);
        if(i_ret < 0)
        {
            printk("[%s] usb_ep_enable event IN fail\n",__func__);
            goto clean_up_data_out;
        }
        ps_ptp->ps_intrpt_ep->driver_data = ps_ptp;

        ps_write = &ps_ptp->s_event_write;
        for(i_cnt = 0; i_cnt < AB_PTP_EVENT_READ_REQ_NUM; i_cnt++)
        {
            ps_req = alloc_ep_req(ps_ptp->ps_intrpt_ep, AB_PTP_EVENT_REQ_LEN);
            if(!ps_req)
            {
                printk("[%s] alloc_ep_req event IN %d fail\n",__func__, i_cnt);
                i_ret = -ENOMEM;
                goto clean_up_event_in;
            }

            ps_list = NULL;
            ps_list = kzalloc(sizeof(struct AB_F_PTP_REQ_LIST_ST), GFP_ATOMIC);
            if(!ps_list)
            {
                printk("[%s] alloc list event IN %d fail\n",__func__, i_cnt);
                free_ep_req(ps_ptp->ps_intrpt_ep, ps_req);
                i_ret = -ENOMEM;
                goto clean_up_event_in;
            }

            ps_list->ps_req = ps_req;
            list_add_tail(&ps_list->s_list, &ps_write->s_req_free);
        }
    }

    return 0;

clean_up_event_in:
    if(ps_ptp->ps_intrpt_ep)
    {
        usb_ep_disable(ps_ptp->ps_intrpt_ep);
        ps_write = &ps_ptp->s_event_write;
        list_for_each_entry_safe(ps_list, ps_next, 
                                &ps_write->s_req_free, s_list)
        {
            free_ep_req(ps_ptp->ps_intrpt_ep, ps_list->ps_req);
            list_del(&ps_list->s_list);
            kfree(ps_list);
        }
    }
clean_up_data_out:
    if(ps_ptp->ps_out_ep)
    {
        usb_ep_disable(ps_ptp->ps_out_ep);
        ps_read = &ps_ptp->s_data_read;
    }
clean_up_data_in:
    if(ps_ptp->ps_in_ep)
    {
        usb_ep_disable(ps_ptp->ps_in_ep);
        ps_write = &ps_ptp->s_data_write;
        list_for_each_entry_safe(ps_list, ps_next, 
                                &ps_write->s_req_free, s_list)
        {
            free_ep_req(ps_ptp->ps_in_ep, ps_list->ps_req);
            list_del(&ps_list->s_list);
            kfree(ps_list);
        }
    }
fail:
    return i_ret;
}

static void ab_ptp_free(struct usb_function *ps_func)
{
    struct AB_F_PTP_ST *ps_ptp = func_to_ab_ptp(ps_func);
    struct AB_F_PTP_OPTS_ST *ps_opts;

    printk("ab_ptp_free\n");

    ps_opts = container_of(ps_func->fi, struct AB_F_PTP_OPTS_ST, s_func_inst);
    ps_opts->i_refcnt -= 1;
    kfree(ps_ptp);
}

static void ab_ptp_unbind(
    struct usb_configuration *ps_config, struct usb_function *ps_func)
{
    struct AB_F_PTP_ST *ps_ptp = func_to_ab_ptp(ps_func);

    INFO(ps_config->cdev, "ab_ptp_unbind\n");

    device_destroy(ps_ptp_class, MKDEV(i_major, ps_ptp->i_data_minor));
    cdev_del(&ps_ptp->s_data_cdev);

    device_destroy(ps_ptp_class, MKDEV(i_major, ps_ptp->i_event_minor));
    cdev_del(&ps_ptp->s_event_cdev);

	usb_free_all_descriptors(ps_func);

    ps_ptp->ps_ctrl_req->complete = NULL;
    ps_ptp->ps_ctrl_req->context = NULL;
    ps_ptp->ps_ctrl_req = NULL;
    ps_ptp->ps_ep0 = NULL;

    ab_ptp_comm_remove_cdev();
}

static int ab_ptp_bind(
    struct usb_configuration *ps_config, struct usb_function *ps_func)
{
    struct usb_composite_dev *ps_cdev = ps_config->cdev;
    struct AB_F_PTP_ST *ps_ptp = func_to_ab_ptp(ps_func);
    struct usb_string *ps_string;
    struct usb_ep *ps_ep;
    struct device *ps_device;
	dev_t e_dev;
    int i_ret = 0;

    INFO(ps_cdev, "ab_ptp_bind\n");

    // alloc string
    ps_string = usb_gstrings_attach(ps_cdev, aps_ptp_func_str, 
                                    ARRAY_SIZE(as_ptp_func_str_def));
    if(IS_ERR(ps_string))
    {
        i_ret = PTR_ERR(ps_string);
        goto fail;
    }
    s_ptp_intf_desc.iInterface = ps_string[AB_PTP_FUNC_STR_ID].id;

    // alloc intf ID
	i_ret = usb_interface_id(ps_config, ps_func);
	if(i_ret < 0)
		goto fail;

	s_ptp_intf_desc.bInterfaceNumber = i_ret;

    // alloc ep
	i_ret = -ENODEV;
	ps_ep = usb_ep_autoconfig(ps_cdev->gadget, &s_ptp_fs_in_ep_desc);
	if(!ps_ep)
		goto fail;

	ps_ptp->ps_in_ep = ps_ep;

	ps_ep = usb_ep_autoconfig(ps_cdev->gadget, &s_ptp_fs_out_ep_desc);
	if(!ps_ep)
		goto fail;

    ps_ptp->ps_out_ep = ps_ep;

	ps_ep = usb_ep_autoconfig(ps_cdev->gadget, &s_ptp_intrpt_ep_desc);
	if(!ps_ep)
		goto fail;

    ps_ptp->ps_intrpt_ep = ps_ep;

    s_ptp_hs_in_ep_desc.bEndpointAddress = 
        s_ptp_fs_in_ep_desc.bEndpointAddress;
    s_ptp_ss_in_ep_desc.bEndpointAddress = 
        s_ptp_fs_in_ep_desc.bEndpointAddress;
    s_ptp_hs_out_ep_desc.bEndpointAddress = 
        s_ptp_fs_out_ep_desc.bEndpointAddress;
    s_ptp_ss_out_ep_desc.bEndpointAddress = 
        s_ptp_fs_out_ep_desc.bEndpointAddress;

	i_ret = usb_assign_descriptors(ps_func, aps_ptp_fs_desc, 
                                    aps_ptp_hs_desc, 
                                    aps_ptp_ss_desc, NULL);
	if(i_ret)
		goto fail;

    // config control request
    ps_ptp->ps_ep0 = ps_cdev->gadget->ep0;
    ps_ptp->ps_ctrl_req = ps_cdev->req;
    ps_ptp->ps_ctrl_req->complete = ab_ptp_ctrl_req_complete;
    ps_ptp->ps_ctrl_req->context = ps_ptp;

    // init file management
    spin_lock_init(&ps_ptp->s_data_write.s_lock);
    INIT_LIST_HEAD(&ps_ptp->s_data_write.s_req_free);
	init_waitqueue_head(&ps_ptp->s_data_write.s_queue);

    spin_lock_init(&ps_ptp->s_data_read.s_lock);
	init_waitqueue_head(&ps_ptp->s_data_read.s_queue);
    INIT_LIST_HEAD(&ps_ptp->s_data_read.s_completed_out_req);

    spin_lock_init(&ps_ptp->s_event_write.s_lock);
    INIT_LIST_HEAD(&ps_ptp->s_event_write.s_req_free);
	init_waitqueue_head(&ps_ptp->s_event_write.s_queue);

	// create char device
	cdev_init(&ps_ptp->s_data_cdev, &ab_f_ptp_data_fops);
	e_dev = MKDEV(i_major, ps_ptp->i_data_minor);
	i_ret = cdev_add(&ps_ptp->s_data_cdev, e_dev, 1);
	if(i_ret)
		goto fail_free_descs;

	ps_device = device_create(ps_ptp_class, NULL, e_dev, NULL,
			                    "ab_ptp%d", ps_ptp->i_data_minor);
	if(IS_ERR(ps_device))
    {
		i_ret = PTR_ERR(ps_device);
        cdev_del(&ps_ptp->s_data_cdev);
		goto fail_free_descs;
	}

	cdev_init(&ps_ptp->s_event_cdev, &ab_f_ptp_event_fops);
	e_dev = MKDEV(i_major, ps_ptp->i_event_minor);
	i_ret = cdev_add(&ps_ptp->s_event_cdev, e_dev, 1);
	if(i_ret)
		goto del_dev;

	ps_device = device_create(ps_ptp_class, NULL, e_dev, NULL,
			                    "ab_ptp%d", ps_ptp->i_event_minor);
	if(IS_ERR(ps_device))
    {
		i_ret = PTR_ERR(ps_device);
        cdev_del(&ps_ptp->s_event_cdev);
		goto del_dev;
	}

    ab_ptp_comm_make_cdev();

    return 0;

del_dev:
    device_destroy(ps_ptp_class, MKDEV(i_major, ps_ptp->i_data_minor));
    cdev_del(&ps_ptp->s_data_cdev);
fail_free_descs:
    ps_ptp->ps_ctrl_req->complete = NULL;
    ps_ptp->ps_ctrl_req->context = NULL;
    ps_ptp->ps_ctrl_req = NULL;
    ps_ptp->ps_ep0 = NULL;
    usb_free_all_descriptors(ps_func);
fail:
	printk("ab_ptp_bind fail\n");
    return i_ret;
}

//===============================================================================
//========                      usb function inst                        ========
//===============================================================================

static inline int ab_ptp_get_minor(void)
{
	int i_ret = 0;

	i_ret = ida_simple_get(&ab_ptp_ida, 0, AB_PTP_MINORS, GFP_KERNEL);
	if(i_ret >= AB_PTP_MINORS)
    {
		ida_simple_remove(&ab_ptp_ida, i_ret);
		i_ret = -ENODEV;
	}

	return i_ret;
}

static inline void ab_ptp_put_minor(int i_minor)
{
	ida_simple_remove(&ab_ptp_ida, i_minor);
}

static void ab_ptp_free_inst(struct usb_function_instance *ps_func_inst)
{
	struct AB_F_PTP_OPTS_ST *ps_opts;

	ps_opts = container_of(ps_func_inst, struct AB_F_PTP_OPTS_ST, s_func_inst);

	mutex_lock(&ab_ptp_ida_lock);
    ab_ptp_put_minor(ps_opts->i_data_minor);
    ab_ptp_put_minor(ps_opts->i_event_minor);
	if(ida_is_empty(&ab_ptp_ida))
		ab_gptp_cleanup();

	mutex_unlock(&ab_ptp_ida_lock);

    ab_ptp_comm_remove_class();

	kfree(ps_opts);
}

static struct usb_function_instance *ab_ptp_alloc_inst(void)
{
	struct AB_F_PTP_OPTS_ST *ps_opts;
	struct usb_function_instance *ps_ret = NULL;
	int i_ret = 0;

	ps_opts = kzalloc(sizeof(*ps_opts), GFP_KERNEL);
	if(!ps_opts)
		return ERR_PTR(-ENOMEM);

	ps_opts->s_func_inst.free_func_inst = ab_ptp_free_inst;
	ps_ret = &ps_opts->s_func_inst;

    ab_ptp_comm_make_class();

	mutex_lock(&ab_ptp_ida_lock);
	if(ida_is_empty(&ab_ptp_ida))
    {
		i_ret = ab_gptp_setup(NULL, AB_PTP_MINORS);
		if(i_ret)
        {
			ps_ret = ERR_PTR(i_ret);
			kfree(ps_opts);
			goto unlock;
		}
	}

	ps_opts->i_data_minor = ab_ptp_get_minor();
	if(ps_opts->i_data_minor < 0)
    {
		ps_ret = ERR_PTR(ps_opts->i_data_minor);
		kfree(ps_opts);
		if(ida_is_empty(&ab_ptp_ida))
			ab_gptp_cleanup();

		goto unlock;
	}

    ps_opts->i_event_minor = ab_ptp_get_minor();
	if(ps_opts->i_event_minor < 0)
    {
        ab_ptp_put_minor(ps_opts->i_data_minor);
		ps_ret = ERR_PTR(ps_opts->i_event_minor);
		kfree(ps_opts);
		if(ida_is_empty(&ab_ptp_ida))
			ab_gptp_cleanup();

		goto unlock;
	}

	config_group_init_type_name(&ps_opts->s_func_inst.group, 
                                "", &s_ptp_func_type);

unlock:
	mutex_unlock(&ab_ptp_ida_lock);
	return ps_ret;
}

static struct usb_function *ab_ptp_alloc(
    struct usb_function_instance *ps_func_inst)
{
	struct AB_F_PTP_ST *ps_ptp;
    struct AB_F_PTP_OPTS_ST *ps_opts;

	ps_ptp = kzalloc(sizeof(*ps_ptp), GFP_KERNEL);
	if(!ps_ptp)
		return ERR_PTR(-ENOMEM);

    ps_opts = container_of(ps_func_inst, struct AB_F_PTP_OPTS_ST, s_func_inst);
    ps_ptp->i_data_minor = ps_opts->i_data_minor;
    ps_ptp->i_event_minor = ps_opts->i_event_minor;
    snprintf(ps_ptp->ac_name, sizeof(ps_ptp->ac_name), "%s", ps_opts->ac_name);
    ps_opts->i_refcnt += 1;

	ps_ptp->s_func.name         = "ab_ptp";
	ps_ptp->s_func.bind         = ab_ptp_bind;
	ps_ptp->s_func.unbind       = ab_ptp_unbind;
	ps_ptp->s_func.free_func    = ab_ptp_free;
	ps_ptp->s_func.set_alt      = ab_ptp_set_alt;
	ps_ptp->s_func.disable      = ab_ptp_disable;
	ps_ptp->s_func.setup        = ab_ptp_setup;

	return &ps_ptp->s_func;
}

DECLARE_USB_FUNCTION_INIT(ab_ptp, ab_ptp_alloc_inst, ab_ptp_alloc);
MODULE_LICENSE("GPL");

int ab_gptp_setup(struct usb_gadget *ps_gadget, int i_count)
{
	dev_t e_dev;
	int i_ret = 0;

	ps_ptp_class = class_create(THIS_MODULE, "ab_ptp");
	if(IS_ERR(ps_ptp_class))
    {
		i_ret = PTR_ERR(ps_ptp_class);
		ps_ptp_class = NULL;
		return i_ret;
	}

	i_ret = alloc_chrdev_region(&e_dev, 0, i_count, "ab_ptp");
	if(i_ret)
    {
		class_destroy(ps_ptp_class);
		ps_ptp_class = NULL;
		return i_ret;
	}

	i_major = MAJOR(e_dev);
	i_minors = i_count;

	return 0;
}

void ab_gptp_cleanup(void)
{
	if(i_major)
    {
		unregister_chrdev_region(MKDEV(i_major, 0), i_minors);
		i_major = i_minors = 0;
	}

	class_destroy(ps_ptp_class);
	ps_ptp_class = NULL;
}

